const { json } = require("sequelize");
const sequelize = require("../db");
const { QueryTypes } = require('sequelize');

const formatDate = (date) => {
    const year = date.getFullYear();
    const month = String(date.getMonth() + 1).padStart(2, '0');
    const day = String(date.getDate()).padStart(2, '0');
    const hours = String(date.getHours()).padStart(2, '0');
    const minutes = String(date.getMinutes()).padStart(2, '0');
    const seconds = String(date.getSeconds()).padStart(2, '0');
    return `${year}-${month}-${day} ${hours}:${minutes}:${seconds}`;
};
// Create item for Shop 1
const CreateShop1Stock = async (req, res) => {
    try {
        // Destructure the result and metadata from the query
        const [results] = await sequelize.query(
            `SELECT * FROM Shop1_Stock WHERE ItemName = :itemName`,
            {
                replacements: {
                    itemName: req.body.itemName,
                },
            }
        );

        const existingRecords = results;

        let dbResults;
        if (existingRecords.length > 0) {
            // Update existing stock
            const [updateResults] = await sequelize.query(
                `UPDATE Shop1_Stock
                SET stock = stock + :stock
                WHERE ItemName = :itemName`,
                {
                    replacements: {
                        stock: parseInt(req.body.stock),
                        itemName: req.body.itemName,
                    },
                }
            );
            dbResults = updateResults;
        } else {
            // Insert new item
            const [insertResults] = await sequelize.query(
                `INSERT INTO Shop1_Stock (
                    ItemId,
                    ItemName,
                    ItemCategoryId,
                    stockIndate,
                    stock
                ) VALUES (
                    :itemId,
                    :itemName,
                    :itemCategoryId,
                    :stockIndate,
                    :stock
                )`,
                {
                    replacements: {
                        itemId: parseInt(req.body.itemId),
                        itemName: req.body.itemName,
                        itemCategoryId: parseInt(req.body.itemCategoryId),
                        stockIndate: req.body.stockIndate,
                        stock: parseInt(req.body.stock),
                    },
                }
            );
            dbResults = insertResults;
        }

        res.json({ message: 'Operation successful', data: dbResults });
    } catch (error) {
        return res.status(400).json({ error: error.message });
    }
};

// Create item for Shop 2
const CreateShop2Stock = async (req, res) => {
    try {
        // Destructure the result and metadata from the query
        const [results, metadata] = await sequelize.query(
            `
            SELECT * FROM Shop2_Stock
            WHERE ItemName = :itemName AND stockExpireDate = :stockExpireDate
            `,
            {
                replacements: {
                    itemName: req.body.itemName,
                    stockExpireDate: req.body.stockExpireDate,
                },
            }
        );

        const existingRecords = results;

        let dbResults;
        if (existingRecords.length > 0) {
            // Update existing stock
            const [updateResults, updateMetadata] = await sequelize.query(
                `
                UPDATE Shop2_Stock
                SET stock = stock + :stock
                WHERE ItemName = :itemName AND stockExpireDate = :stockExpireDate
                `,
                {
                    replacements: {
                        stock: parseInt(req.body.stock),
                        itemName: req.body.itemName,
                        stockExpireDate: req.body.stockExpireDate,
                    },
                }
            );
            dbResults = updateResults;
        } else {
            // Insert new item
            const [insertResults, insertMetadata] = await sequelize.query(
                `
                INSERT INTO Shop2_Stock (
                    ItemId,
                    ItemName,
                    ItemCategoryId,
                    stockIndate,
                    stockExpireDate,
                    stock
                ) VALUES (
                    :itemId,
                    :itemName,
                    :itemCategoryId,
                    :stockIndate,
                    :stockExpireDate,
                    :stock
                )
                `,
                {
                    replacements: {
                        itemId: parseInt(req.body.itemId),
                        itemName: req.body.itemName,
                        itemCategoryId: parseInt(req.body.itemCategoryId),
                        stockIndate: req.body.stockIndate,
                        stockExpireDate: req.body.stockExpireDate,
                        stock: parseInt(req.body.stock)
                    },
                }
            );
            dbResults = insertResults;
        }
        res.json({ message: 'Operation successful', data: dbResults });
    } catch (error) {
        return res.status(400).json({ error: error.message });
    }
};

// Create item for Shop 3
const CreateShop3Stock = async (req, res) => {
    try {
        // Destructure the result and metadata from the query
        const [results, metadata] = await sequelize.query(
            `
            SELECT * FROM Shop3_Stock
            WHERE ItemName = :itemName AND stockExpireDate = :stockExpireDate
            `,
            {
                replacements: {
                    itemName: req.body.itemName,
                    stockExpireDate: req.body.stockExpireDate,
                },
            }
        );

        const existingRecords = results;

        let dbResults;
        if (existingRecords.length > 0) {
            // Update existing stock
            const [updateResults, updateMetadata] = await sequelize.query(
                `
                UPDATE Shop3_Stock
                SET stock = stock + :stock
                WHERE ItemName = :itemName AND stockExpireDate = :stockExpireDate
                `,
                {
                    replacements: {
                        stock: parseInt(req.body.stock),
                        itemName: req.body.itemName,
                        stockExpireDate: req.body.stockExpireDate,
                    },
                }
            );
            dbResults = updateResults;
        } else {
            // Insert new item
            const [insertResults, insertMetadata] = await sequelize.query(
                `
                INSERT INTO Shop3_Stock (
                    ItemId,
                    ItemName,
                    ItemCategoryId,
                    stockIndate,
                    stockExpireDate,
                    stock
                ) VALUES (
                    :itemId,
                    :itemName,
                    :itemCategoryId,
                    :stockIndate,
                    :stockExpireDate,
                    :stock
                )
                `,
                {
                    replacements: {
                        itemId: parseInt(req.body.itemId),
                        itemName: req.body.itemName,
                        itemCategoryId: parseInt(req.body.itemCategoryId),
                        stockIndate: req.body.stockIndate,
                        stockExpireDate: req.body.stockExpireDate,
                        stock: parseInt(req.body.stock)
                    },
                }
            );
            dbResults = insertResults;
        }
        res.json({ message: 'Operation successful', data: dbResults });
    } catch (error) {
        return res.status(400).json({ error: error.message });
    }
};

//main stock
const getStockShop1 = async (req, res) => {
    try {
        const [results, metadata] = await sequelize.query(`
            SELECT stockId,itemid,itemname,itemcategoryid, stockindate,stock 
            FROM Shop1_Stock where stock > 0
            order by stockindate desc
        `);
        res.json(results);
    } catch (error) {
        return res.status(400).json({ error: error.message });
    }
}

const getStoreStock = async (req, res) => {
    try {
        const [results, metadata] = await sequelize.query(`
            SELECT stockId,itemid,itemname,itemcategoryid, stockindate,stockexpiredate,stock,laststockoutdate 
            FROM Shop1_Stock 
            where stock != 0
            order by stockindate desc
        `);
        res.json(results);
    } catch (error) {
        return res.status(400).json({ error: error.message });
    }
}

const getStockShop2 = async (req, res) => {
    try {
        const [results, metadata] = await sequelize.query(`
                SELECT stockId,itemid,itemname,itemcategoryid, stockindate,stockexpiredate,stock,laststockoutdate 
                FROM Shop2_Stock
                where stock != 0
                order by stockindate desc
                `);
        res.json(results);
    } catch (error) {
        return res.status(400).json({ error: error.message });
    }
}

const getStockShop3 = async (req, res) => {
    try {
        const [results, metadata] = await sequelize.query(`
            SELECT stockId,itemid,itemname,itemcategoryid, stockindate,stockexpiredate,stock,laststockoutdate 
            FROM Shop3_Stock
            where stock != 0
            order by stockindate desc`);
        res.json(results);
    } catch (error) {
        return res.status(400).json({ error: error.message });
    }
}
const getStockShop4 = async (req, res) => {
    try {
        const [results, metadata] = await sequelize.query(`
            SELECT stockId,itemid,itemname,itemcategoryid, stockindate,stockexpiredate,stock,laststockoutdate 
            FROM Shop4_Stock
            where stock != 0
            order by stockindate desc`);
        res.json(results);
    } catch (error) {
        return res.status(400).json({ error: error.message });
    }
}



const DeleteShop1Stock = async (req, res) => {
    try {
        const stockid = req.body.stockid; 
        const [results, metadata] = await sequelize.query(`
            DELETE FROM shop1_stock WHERE stockid = :stockid
        `, {
            replacements: { stockid: stockid }
        });
        if (results.affectedRows === 0) { 
            res.status(404).json({ message: "stock not found." });
        } else {
            res.json({ message: "stock deleted successfully." });
        }
    } catch (error) {
        return res.status(400).json({ error: error.message });
    }
}

const DeleteShop2Stock = async (req, res) => {
    try {
        const stockid = req.body.stockid; 
        const [results, metadata] = await sequelize.query(`
            DELETE FROM shop2_stock WHERE stockid = :stockid
        `, {
            replacements: { stockid: stockid }
        });
        if (results.affectedRows === 0) { 
            res.status(404).json({ message: "stock not found." });
        } else {
            res.json({ message: "stock deleted successfully." });
        }
    } catch (error) {
        return res.status(400).json({ error: error.message });
    }
}

const DeleteShop3Stock = async (req, res) => {
    try {
        const stockid = req.body.stockid; 
        const [results, metadata] = await sequelize.query(`
            DELETE FROM shop3_stock WHERE stockid = :stockid
        `, {
            replacements: { stockid: stockid }
        });
        if (results.affectedRows === 0) { 
            res.status(404).json({ message: "stock not found." });
        } else {
            res.json({ message: "stock deleted successfully." });
        }
    } catch (error) {
        return res.status(400).json({ error: error.message });
    }
}


const BillShop1Items = async (req, res) => {
    try {
        const items = req.body.items; 
        
        for (const item of items) {
            let { itemName, qty, itemId, itemCategoryId } = item; 
            qty = parseInt(qty);

            while (qty > 0) {
                
                const [stocks] = await sequelize.query(`
                    SELECT * FROM Shop1_Stock
                    WHERE ItemName = :itemName AND stock > 0
                    LIMIT 1
                `, {
                    replacements: {
                        itemName
                    }
                });

                if (stocks.length > 0) {
                    // Existing stock handling
                    const stockItem = stocks[0];
                    const availableStock = stockItem.stock;
                    if (availableStock >= qty) {
                        // Deduct the required qty from available stock
                        await sequelize.query(`
                            UPDATE Shop1_Stock
                            SET stock = stock - :qty
                            WHERE stockId = :stockId
                        `, {
                            replacements: {
                                qty,
                                todayDate: new Date(), 
                                stockId: stockItem.stockId
                            }
                        });
                        qty = 0;
                    } else {
                        // Set stock to zero and continue deducting the remaining qty
                        await sequelize.query(`
                            UPDATE Shop1_Stock
                            SET stock = 0
                            WHERE stockId = :stockId
                        `, {
                            replacements: {
                                todayDate: new Date(),
                                stockId: stockItem.stockId
                            }
                        });
                        qty -= availableStock;
                    }
                } else {
                    // No stock with positive quantity found
                    const [maxStocks] = await sequelize.query(`
                        SELECT * FROM Shop1_Stock
                        WHERE ItemName = :itemName
                        LIMIT 1
                    `, {
                        replacements: {
                            itemName
                        }
                    });

                    if (maxStocks.length === 0) {
                        // Item not found in stock table, insert new record with negative stock
                        await sequelize.query(`
                            INSERT INTO Shop1_Stock (ItemId, ItemName, ItemCategoryId, stock)
                            VALUES (:itemId, :itemName, :itemCategoryId, :negativeQty)
                        `, {
                            replacements: {
                                itemId,
                                itemName,
                                itemCategoryId,
                                negativeQty: -qty,
                            }
                        });
                        qty = 0;
                    } else {
                        // Item exists but with zero stock, deduct negative stock
                        const stockItem = maxStocks[0];
                        await sequelize.query(`
                            UPDATE Shop1_Stock
                            SET stock = stock - :qty
                            WHERE stockId = :stockId
                        `, {
                            replacements: {
                                qty,
                                todayDate: new Date(),
                                stockId: stockItem.stockId
                            }
                        });
                        qty = 0;
                    }
                }
            }
        }
        res.json({ message: 'Billing successful' });
    } catch (error) {
        return res.status(400).json({ error: error.message });
    }
};

const BillShop2Items = async (req, res) => {
    try {
        const items = req.body.items; 
        
        for (const item of items) {
            let { itemName, qty, itemId, itemCategoryId } = item; 
            qty = parseInt(qty);

            while (qty > 0) {
                
                const [stocks] = await sequelize.query(`
                    SELECT * FROM Shop2_Stock
                    WHERE ItemName = :itemName AND stock > 0
                    ORDER BY stockExpireDate ASC
                    LIMIT 1
                `, {
                    replacements: {
                        itemName
                    }
                });

                if (stocks.length > 0) {
                    // Existing stock handling
                    const stockItem = stocks[0];
                    const availableStock = stockItem.stock;
                    if (availableStock >= qty) {
                        // Deduct the required qty from available stock
                        await sequelize.query(`
                            UPDATE Shop2_Stock
                            SET stock = stock - :qty, lastStockOutDate = :todayDate
                            WHERE stockId = :stockId
                        `, {
                            replacements: {
                                qty,
                                todayDate: new Date(), 
                                stockId: stockItem.stockId
                            }
                        });
                        qty = 0;
                    } else {
                        // Set stock to zero and continue deducting the remaining qty
                        await sequelize.query(`
                            UPDATE Shop2_Stock
                            SET stock = 0, lastStockOutDate = :todayDate
                            WHERE stockId = :stockId
                        `, {
                            replacements: {
                                todayDate: new Date(),
                                stockId: stockItem.stockId
                            }
                        });
                        qty -= availableStock;
                    }
                } else {
                    // No stock with positive quantity found
                    const [maxStocks] = await sequelize.query(`
                        SELECT * FROM Shop2_Stock
                        WHERE ItemName = :itemName
                        ORDER BY stockExpireDate DESC
                        LIMIT 1
                    `, {
                        replacements: {
                            itemName
                        }
                    });

                    if (maxStocks.length === 0) {
                        // Item not found in stock table, insert new record with negative stock
                        await sequelize.query(`
                            INSERT INTO Shop2_Stock (ItemId, ItemName, ItemCategoryId, stock, lastStockOutDate)
                            VALUES (:itemId, :itemName, :itemCategoryId, :negativeQty, :todayDate)
                        `, {
                            replacements: {
                                itemId,
                                itemName,
                                itemCategoryId,
                                negativeQty: -qty,
                                todayDate: new Date()
                            }
                        });
                        qty = 0;
                    } else {
                        // Item exists but with zero stock, deduct negative stock
                        const stockItem = maxStocks[0];
                        await sequelize.query(`
                            UPDATE Shop2_Stock
                            SET stock = stock - :qty, lastStockOutDate = :todayDate
                            WHERE stockId = :stockId
                        `, {
                            replacements: {
                                qty,
                                todayDate: new Date(),
                                stockId: stockItem.stockId
                            }
                        });
                        qty = 0;
                    }
                }
            }
        }
        res.json({ message: 'Billing successful' });
    } catch (error) {
        return res.status(400).json({ error: error.message });
    }
};

const BillShop3Items = async (req, res) => {
    try {
        const items = req.body.items; 
        
        for (const item of items) {
            let { itemName, qty, itemId, itemCategoryId } = item; 
            qty = parseInt(qty);

            while (qty > 0) {
                
                const [stocks] = await sequelize.query(`
                    SELECT * FROM Shop3_Stock
                    WHERE ItemName = :itemName AND stock > 0
                    ORDER BY stockExpireDate ASC
                    LIMIT 1
                `, {
                    replacements: {
                        itemName
                    }
                });

                if (stocks.length > 0) {
                    // Existing stock handling
                    const stockItem = stocks[0];
                    const availableStock = stockItem.stock;
                    if (availableStock >= qty) {
                        // Deduct the required qty from available stock
                        await sequelize.query(`
                            UPDATE Shop3_Stock
                            SET stock = stock - :qty, lastStockOutDate = :todayDate
                            WHERE stockId = :stockId
                        `, {
                            replacements: {
                                qty,
                                todayDate: new Date(), 
                                stockId: stockItem.stockId
                            }
                        });
                        qty = 0;
                    } else {
                        // Set stock to zero and continue deducting the remaining qty
                        await sequelize.query(`
                            UPDATE Shop3_Stock
                            SET stock = 0, lastStockOutDate = :todayDate
                            WHERE stockId = :stockId
                        `, {
                            replacements: {
                                todayDate: new Date(),
                                stockId: stockItem.stockId
                            }
                        });
                        qty -= availableStock;
                    }
                } else {
                    // No stock with positive quantity found
                    const [maxStocks] = await sequelize.query(`
                        SELECT * FROM Shop3_Stock
                        WHERE ItemName = :itemName
                        ORDER BY stockExpireDate DESC
                        LIMIT 1
                    `, {
                        replacements: {
                            itemName
                        }
                    });

                    if (maxStocks.length === 0) {
                        // Item not found in stock table, insert new record with negative stock
                        await sequelize.query(`
                            INSERT INTO Shop3_Stock (ItemId, ItemName, ItemCategoryId, stock, lastStockOutDate)
                            VALUES (:itemId, :itemName, :itemCategoryId, :negativeQty, :todayDate)
                        `, {
                            replacements: {
                                itemId,
                                itemName,
                                itemCategoryId,
                                negativeQty: -qty,
                                todayDate: new Date()
                            }
                        });
                        qty = 0;
                    } else {
                        // Item exists but with zero stock, deduct negative stock
                        const stockItem = maxStocks[0];
                        await sequelize.query(`
                            UPDATE Shop3_Stock
                            SET stock = stock - :qty, lastStockOutDate = :todayDate
                            WHERE stockId = :stockId
                        `, {
                            replacements: {
                                qty,
                                todayDate: new Date(),
                                stockId: stockItem.stockId
                            }
                        });
                        qty = 0;
                    }
                }
            }
        }
        res.json({ message: 'Billing successful' });
    } catch (error) {
        return res.status(400).json({ error: error.message });
    }
};






// Stock allocation function
const allocateStockSimple = async (req, res) => {
    const { fromShop, targetShop } = req.body;
    const { itemName, qty, expirationDate } = req.body;
    const quantity = parseInt(qty);
  
    console.log(req.body);
  
    if (!itemName || !quantity || !targetShop) {
      return res.status(400).json({ error: 'Item name, quantity, and target shop are required.' });
    }
  
    try {
      await sequelize.transaction(async (t) => {
        // Deduct stock from Shop1
        
        if (targetShop === 'shop2' || targetShop === 'shop3' || targetShop === 'shop4') {
            const shop1StockItem = await deductStockFromShop1Simple(itemName, quantity, expirationDate, t);
          await addStockToTargetShopSimple(targetShop, itemName, quantity, shop1StockItem, t);
        }
        else if (targetShop === 'shop1') {
          console.log("expirationDate", expirationDate);
          const stockItem = await deductfromtarget(itemName, fromShop, quantity, expirationDate, t);
          await addStocktoMainStock(targetShop, itemName, quantity, stockItem, t);
          // Removed: await t.commit();
        }
        else {
          throw new Error('Invalid target shop.');
        }
      });
  
      res.json({ message: 'Stock allocated successfully.' });
    } catch (error) {
      console.error('Error allocating stock:', error);
      res.status(400).json({ error: error.message });
    }
  };
  

const deductStockFromShop1Simple = async (itemName, qty,expirationDate, transaction) => {

  const stocks = await sequelize.query(
    `
    SELECT * FROM Shop1_Stock
    WHERE ItemName = :itemName AND stock > 0 AND stockExpireDate = :expirationDate
    `,
    {
      replacements: { itemName,expirationDate },
      transaction,
      type: QueryTypes.SELECT,
    }
  );

  if (stocks.length === 0) {
    throw new Error(`Insufficient stock in Shop1 for item ${itemName}`);
  }

  const stockItem = stocks[0];
  const availableStock = stockItem.stock;

  if (availableStock < qty) {
    throw new Error(`Insufficient stock in Shop1 for item ${itemName}. Available: ${availableStock}, Required: ${qty}`);
  }


  await sequelize.query(
    `
    UPDATE Shop1_Stock
    SET stock = stock - :qty, laststockoutdate = :todayDate
    WHERE stockId = :id
    `,
    {
      replacements: {
        qty,
        todayDate: new Date(),
        id: stockItem.stockId,
      },
      transaction,
      type: QueryTypes.UPDATE,
    }
  );

  return stockItem; 
};


const addStockToTargetShopSimple = async (targetShop, itemName, qty, stockItem, transaction) => {
    
    const targetShopTable = targetShop === 'shop2' ? 'Shop2_Stock' : targetShop ==='shop4' ? 'Shop4_Stock' : 'Shop3_Stock';
    
    
    const existingRecords = await sequelize.query(
        `
        SELECT * FROM ${targetShopTable}
        WHERE ItemName = :itemName AND stockExpireDate = :stockExpireDate
        `,
        {
            replacements: {
                itemName,
                stockExpireDate: stockItem.stockExpireDate,
      },
      transaction,
      type: QueryTypes.SELECT,
    }
);

if (existingRecords.length > 0) {
    // Update existing stock
    await sequelize.query(
        `
        UPDATE ${targetShopTable}
        SET stock = stock + :qty
        WHERE stockid = :id
        `,
        {
            replacements: {
          qty,
          id: existingRecords[0].stockId,
        },
        transaction,
        type: QueryTypes.UPDATE,
    }
);
} else {
    // Insert new stock entry
    //console.log(stockItem)
    await sequelize.query(
        `
        INSERT INTO ${targetShopTable} (
            ItemId,
            ItemName,
            ItemCategoryId,
            stockIndate,
            stockExpireDate,
            stock,
            laststockoutdate
            ) VALUES (
                :itemId,
                :itemName,
                :itemCategoryId,
                :stockIndate,
                :stockExpireDate,
                :stock,
                :laststockoutdate
                )
                `,
                {
        replacements: {
          itemId: stockItem.itemId,
          itemName: stockItem.itemName,
          itemCategoryId: stockItem.itemCategoryId,
          stockIndate: new Date(),
          stockExpireDate: stockItem.stockExpireDate,
          stock: qty,
          laststockoutdate: null,
        },
        transaction,
        type: QueryTypes.INSERT,
    }
);
}
};


//Return Stock functiuons


const deductfromtarget = async (itemName, fromShop, qty, expirationDate, transaction) => {
    const formattedExpirationDate = expirationDate.split(' ')[0];
  
    const stocks = await sequelize.query(
      `
      SELECT * FROM ${fromShop}
      WHERE ItemName = :itemName AND stock > 0 AND stockExpireDate = :formattedExpirationDate
      `,
      {
        replacements: { itemName, formattedExpirationDate },
        transaction,
        type: QueryTypes.SELECT,
      }
    );
  
    if (stocks.length === 0) {
      throw new Error(`Insufficient stock in ${fromShop} for item ${itemName}`);
    }
  
    const stockItem = stocks[0];
    const availableStock = stockItem.stock;
  
    if (availableStock < qty) {
      throw new Error(`Insufficient stock in ${fromShop} for item ${itemName}. Available: ${availableStock}, Required: ${qty}`);
    }
  
    await sequelize.query(
      `
      UPDATE ${fromShop}
      SET stock = stock - :qty, laststockoutdate = :todayDate
      WHERE stockId = :id
      `,
      {
        replacements: {
          qty,
          todayDate: new Date(),
          id: stockItem.stockId,
        },
        transaction,
        type: QueryTypes.UPDATE,
      }
    );
  
    return stockItem; 
  };
  
  
  const addStocktoMainStock = async (targetShop, itemName, qty, stockItem, transaction) => {
    const targetShopTable = 'Shop1_Stock';
    //console.log("stockItem", stockItem);
  
    const existingRecords = await sequelize.query(
      `
      SELECT * FROM ${targetShopTable}
      WHERE ItemName = :itemName AND stockExpireDate = :expirationDate
      `,
      {
        replacements: {
          itemName,
          expirationDate: stockItem.stockExpireDate,
        },
        transaction,
        type: QueryTypes.SELECT,
      }
    );
  
    //console.log("existingRecords", existingRecords);
  
    if (existingRecords.length > 0) {
      // Update existing stock
      //console.log('Updating existing stock record');
      const result = await sequelize.query(
        `
        UPDATE ${targetShopTable}
        SET stock = stock + :qty
        WHERE stockId = :id
        `,
        {
          replacements: {
            qty,
            id: existingRecords[0].stockId,  // Ensure correct stockId is used
          },
          transaction,
          type: QueryTypes.UPDATE,
        }
      );
  
      //console.log('Rows affected by update:', result[1]);  // Log number of rows affected
    } else {
      // Insert new stock entry
      //console.log('Inserting new stock record');
      await sequelize.query(
        `
        INSERT INTO ${targetShopTable} (
          ItemId,
          ItemName,
          ItemCategoryId,
          stockInDate,
          stockExpireDate,
          stock,
          lastStockOutDate
        ) VALUES (
          :itemId,
          :itemName,
          :itemCategoryId,
          :stockInDate,
          :stockExpireDate,
          :stock,
          :lastStockOutDate
        )
        `,
        {
          replacements: {
            itemId: stockItem.itemId,
            itemName: stockItem.itemName,
            itemCategoryId: stockItem.itemCategoryId,
            stockInDate: new Date(),
            stockExpireDate: stockItem.stockExpireDate,
            stock: qty,
            lastStockOutDate: null,
          },
          transaction,
          type: QueryTypes.INSERT,
        }
      );
    }
  };

  const returnSupplier = async (req, res) => {
    try {
        const { 
            itemName, 
            itemId, 
            ReturnUnitPrice, 
            returnqty, 
            returnValue, 
            SupplierId, 
            returndate,
            stockId
        } = req.body;

        //////console.log(req.body);

        // Insert data into supplier_returns table
        const [insertResults, insertMetadata] = await sequelize.query(
            `
            INSERT INTO supplier_returns (
                itemName,
                itemId,
                ReturnUnitPrice,
                returnqty,
                returnValue,
                SupplierId,
                returndate
            ) VALUES (
                :itemName,
                :itemId,
                :ReturnUnitPrice,
                :returnqty,
                :returnValue,
                :SupplierId,
                :returndate
            )
            `,
            {
                replacements: {
                    itemName: itemName,
                    itemId: parseInt(itemId),
                    ReturnUnitPrice: parseFloat(ReturnUnitPrice),
                    returnqty: parseInt(returnqty),
                    returnValue: parseFloat(returnValue),
                    SupplierId: parseInt(SupplierId),
                    returndate: returndate
                },
            }
        );

        // Update stock in Shop1_Stock table by decrementing the stock using stockId
        const [updateResults, updateMetadata] = await sequelize.query(
            `
            UPDATE Shop1_Stock
            SET stock = stock - :returnqty
            WHERE stockId = :stockId
            `,
            {
                replacements: {
                    returnqty: parseInt(returnqty),
                    stockId: parseInt(stockId)
                },
            }
        );

        // Check if the update affected any rows
        if (updateResults[0] === 0) {
            throw new Error('Stock update failed or stockId not found');
        }

        // Send response after both insert and update
        res.status(200).json({
            message: "Return Supplier data inserted and stock updated successfully",
            insertResults,
            insertMetadata,
            updateResults,
            updateMetadata
        });

    } catch (error) {
        console.error('Error in returnSupplier:', error);
        res.status(500).json({
            message: "Error processing return supplier data and updating stock",
            error: error.message
        });
    }
};

  const editStock = async (req, res) => {
    try {
        const { 
            stockId, 
            returnqty, 
        } = req.body;

        //console.log(req.body);

        // Update stock in Shop1_Stock table by decrementing the stock using stockId
        const [updateResults, updateMetadata] = await sequelize.query(
            `
            UPDATE Shop1_Stock
            SET stock = :returnqty
            WHERE stockId = :stockId
            `,
            {
                replacements: {
                    returnqty: parseInt(returnqty),
                    stockId: parseInt(stockId)
                },
            }
        );

        // Check if the update affected any rows
        if (updateResults[0] === 0) {
            throw new Error('Stock update failed or stockId not found');
        }

        // Send response after both insert and update
        res.status(200).json({
            message: "Return Supplier data inserted and stock updated successfully",
            updateResults,
            updateMetadata
        });

    } catch (error) {
        console.error('Error in edit Stock:', error);
        res.status(500).json({
            message: "Error processing updating stock",
            error: error.message
        });
    }
};



module.exports = {
    CreateShop1Stock,
    CreateShop2Stock,
    CreateShop3Stock,
    getStockShop1,
    getStockShop2,
    getStockShop3,
    DeleteShop1Stock,
    DeleteShop2Stock,
    DeleteShop3Stock,
    BillShop1Items,
    BillShop2Items,
    BillShop3Items,
    allocateStockSimple,
    getStoreStock,
    getStockShop4,
    returnSupplier,
    editStock
}