const { json } = require("sequelize");
const sequelize = require("../db");
const path = require('path');
const fs = require('fs');



// CreateTour: now also handles route_image + route_description
const CreateTour = async (req, res) => {
  const t = await sequelize.transaction();  // Start a transaction

  try {
    const { title, days, description, tourID, rate, route_description } = req.body;
    const dayCountInt = parseInt(days, 10) || 0;

    // Basic guard (optional)
    if (!tourID || !title || !description || Number.isNaN(dayCountInt)) {
      return res.status(400).json({ error: 'Missing or invalid fields.' });
    }

    // Create upload folder if not exists (outside DB transaction)
    const uploadDir = path.join(__dirname, '..', 'Assets', 'Tours');
    if (!fs.existsSync(uploadDir)) {
      fs.mkdirSync(uploadDir, { recursive: true });
    }

    // --- Route image handling (optional) ---
    const routeImageFile = req.files?.route_image;
    let routeSavedFileName = 'no_image.jpg';
    if (routeImageFile) {
      // keep original extension if present, fallback to .jpg
      const ext = (path.extname(routeImageFile.name) || '.jpg').toLowerCase();
      // very light extension clamp (optional)
      const allowed = new Set(['.jpg', '.jpeg', '.png', '.webp']);
      const safeExt = allowed.has(ext) ? ext : '.jpg';

      routeSavedFileName = `tour_${tourID}_route_${Date.now()}${safeExt}`;
      const routeFullPath = path.join(uploadDir, routeSavedFileName);
      await routeImageFile.mv(routeFullPath);
    }

    // Insert TourHeader (now with RoutePhoto + RouteDescription) inside the transaction
    const [headerResult, headerMeta] = await sequelize.query(`
      INSERT INTO TourHeader (TourId, Title, Days, Description, rate, Route_Photo, Route_Description, CreatedAt)
      VALUES (:tourID, :title, :days, :description, :rate, :routePhoto, :routeDescription, NOW())
    `, {
      replacements: {
        tourID,
        title,
        days: dayCountInt,
        description,
        rate,
        routePhoto: routeSavedFileName,
        routeDescription: route_description || null,
      },
      transaction: t
    });

    const savedDetails = [];

    // Loop through each day and insert into TourDetails inside the transaction
    for (let i = 1; i <= dayCountInt; i++) {
      const photoKey = `day_${i}_photo`;
      const descKey = `day_${i}_description`;

      const photoFile = req.files?.[photoKey];
      const dayDescription = req.body[descKey];

      let savedFileName = 'no_image.jpg';
      if (photoFile) {
        const ext = (path.extname(photoFile.name) || '.jpg').toLowerCase();
        const allowed = new Set(['.jpg', '.jpeg', '.png', '.webp']);
        const safeExt = allowed.has(ext) ? ext : '.jpg';

        savedFileName = `tour_${tourID}_day_${i}_${Date.now()}${safeExt}`;
        const fullPath = path.join(uploadDir, savedFileName);
        await photoFile.mv(fullPath);
      }

      const [detailResult, detailMeta] = await sequelize.query(`
        INSERT INTO TourDetails (TourID, DayNumber, Photo, Description, CreatedAt)
        VALUES (:tourID, :dayNumber, :photo, :description, NOW())
      `, {
        replacements: {
          tourID,
          dayNumber: i,
          photo: savedFileName,
          description: dayDescription,
        },
        transaction: t
      });

      savedDetails.push({ day: i, result: detailResult });
    }

    await t.commit(); // Commit the transaction if all inserts succeed

    return res.status(200).json({
      message: 'Tour created successfully.',
      tourID,
      header: headerResult,
      details: savedDetails,
      route: {
        photo: routeSavedFileName,
        description: route_description || null,
      }
    });
  } catch (error) {
    await t.rollback(); // Rollback if any error occurs
    console.error('CreateTour error:', error);
    return res.status(500).json({ error: error.message });
  }
};


const getTheNextId = async (req, res) => {
  try {
    const [result, metadata] = await sequelize.query(`
      SELECT MAX(TourId) AS MaxTourId FROM TourHeader
    `);

    const nextId = result[0].MaxTourId ? result[0].MaxTourId + 1 : 1;
    // return nextId;
     return res.status(200).json({nextId : nextId});
  } catch (error) {
    console.error('Error fetching next TourId:', error);
    throw error;
  }
};

const GetAllTourDetails = async (req, res) => {
  try {
    const [result, metadata] = await sequelize.query(`
      SELECT h.TourId, h.Title,h.days,h.description as CoverDescription,h.rate,d.Photo,d.Description dayDescription
      FROM TourDetails d 
      left join TourHeader h on d.TourId = h.TourId;
    `);

     return res.status(200).json({
      data: result
    });
    
  } catch (error) {
      console.error('Error fetching tour details:', error); 
      return res.status(500).json({ error: error.message });
  }
}
const GetAllTourDetailsById = async (req, res) => {

  const { tourID } = req.body;
  if (!tourID) {
    return res.status(400).json({ error: 'Tour ID is required' });
  }
  try {

        const [detailResult, detailMeta] = await sequelize.query(`
      SELECT h.TourId, h.Title,h.days,h.description as CoverDescription,h.rate,Route_Photo, Route_Description,d.DayNumber,d.Photo,d.Description dayDescription
      FROM TourDetails d 
      left join TourHeader h on d.TourId = h.TourId
      where h.TourId = :tourID;
      `, {
        replacements: {
          tourID,
          
        }
      });
     return res.status(200).json({
      data: detailResult
    });
    
  } catch (error) {
      console.error('Error fetching tour details:', error);
      return res.status(500).json({ error: error.message });
  }
}
const GetAllTourHeader = async (req, res) => {
  try {
    const [result, metadata] = await sequelize.query(`
      SELECT h.id,h.TourId, h.Title,h.days,h.description,h.rate,d.Photo,Route_Photo, Route_Description
      FROM TourHeader h 
      left join ( select tourId, min(Photo) as Photo
      from TourDetails 
      group by TourId) d on d.TourId = h.TourId;
    `);

     return res.status(200).json({
      data: result
    });
    
  } catch (error) {
      console.error('Error fetching tour details:', error);
      return res.status(500).json({ error: error.message });
  }
}



const deleteTour = async (req, res) => {
  const { tourId } = req.body; // Assuming you are passing the tourId in the request body

  try {
    // Start a transaction to ensure both delete operations are executed atomically
    const result = await sequelize.transaction(async (t) => {
      // First, delete from TourDetails where TourId matches
      const detailsDeleted = await sequelize.query(
        `DELETE FROM TourDetails WHERE TourId = :tourId`,
        {
          replacements: { tourId }, // Bind the tourId parameter
          type: sequelize.QueryTypes.DELETE,
          transaction: t,
        }
      );

      // Then, delete from TourHeader where TourId matches
      const headerDeleted = await sequelize.query(
        `DELETE FROM TourHeader WHERE TourId = :tourId`,
        {
          replacements: { tourId }, // Bind the tourId parameter
          type: sequelize.QueryTypes.DELETE,
          transaction: t,
        }
      );

      // Return the result of both delete operations
      return { detailsDeleted, headerDeleted };
    });

    // If no rows were deleted, return a not found response
    if (result.detailsDeleted === 0 && result.headerDeleted === 0) {
      return res.status(404).json({ message: "Tour not found" });
    }

    // Return success response if both deletions were successful
    return res.status(200).json({
      message: "Tour deleted successfully",
    });
    
  } catch (error) {
    console.error('Error deleting tour:', error);
    return res.status(500).json({ error: error.message });
  }
};



const UpdateTour = async (req, res) => {
  const { tourID, title, description, days, rate, inputs, route_description } = req.body;
  const t = await sequelize.transaction();

  try {
    if (!inputs || typeof inputs !== 'string') {
      return res.status(400).json({ error: "Inputs array is missing or invalid" });
    }
    let parsedInputs;
    try {
      parsedInputs = JSON.parse(inputs);
    } catch (e) {
      return res.status(400).json({ error: "Invalid inputs format" });
    }
    if (!Array.isArray(parsedInputs)) {
      return res.status(400).json({ error: "Inputs is not a valid array" });
    }

    // Ensure upload folder exists
    const uploadDir = path.join(__dirname, '..', 'Assets', 'Tours');
    if (!fs.existsSync(uploadDir)) {
      fs.mkdirSync(uploadDir, { recursive: true });
    }

    // --- Optional route image handling ---
    let routePhotoFileName = null;
    const routeImageFile = req.files?.route_image;
    if (routeImageFile) {
      const ext = (path.extname(routeImageFile.name) || '.jpg').toLowerCase();
      const allowed = new Set(['.jpg', '.jpeg', '.png', '.webp']);
      const safeExt = allowed.has(ext) ? ext : '.jpg';
      routePhotoFileName = `tour_${tourID}_route_${Date.now()}${safeExt}`;
      await routeImageFile.mv(path.join(uploadDir, routePhotoFileName));
    }

    // Update TourHeader (keep existing route fields if not provided)
    await sequelize.query(`
      UPDATE TourHeader
         SET Title = :title,
             Description = :description,
             rate = :rate,
             Days = :days,
             Route_Photo = COALESCE(:routePhoto, Route_Photo),
             Route_Description = COALESCE(:routeDescription, Route_Description)
       WHERE TourId = :tourID
    `, {
      replacements: {
        tourID,
        title,
        description,
        rate,
        days: parseInt(days, 10) || 0,
        routePhoto: routePhotoFileName,                  // null -> keep existing
        routeDescription: route_description || null      // null -> keep existing
      },
      transaction: t
    });

    // Update or insert each day's details
    for (let i = 0; i < parsedInputs.length; i++) {
      const dayNumber = i + 1;
      const { description: dayDescription, isFromServer, photo } = parsedInputs[i];
      if (!dayDescription) continue;

      let savedFileName = null;
      const photoFile = req.files?.[`day_${dayNumber}_photo`];

      if (photoFile && !isFromServer) {
        const ext = (path.extname(photoFile.name) || '.jpg').toLowerCase();
        const allowed = new Set(['.jpg', '.jpeg', '.png', '.webp']);
        const safeExt = allowed.has(ext) ? ext : '.jpg';
        savedFileName = `tour_${tourID}_day_${dayNumber}_${Date.now()}${safeExt}`;
        await photoFile.mv(path.join(uploadDir, savedFileName));
      } else if (isFromServer && photo) {
        // keep server-side existing image
        savedFileName = photo.split('/').pop();
      }
      // If savedFileName stays null, COALESCE in SQL will keep current Photo

      // Try UPDATE first (keep existing photo if no new one provided)
      const [_, meta] = await sequelize.query(`
        UPDATE TourDetails
           SET Description = :dayDescription,
               Photo = COALESCE(:photo, Photo)
         WHERE TourId = :tourID AND DayNumber = :dayNumber
      `, {
        replacements: {
          dayDescription,
          photo: savedFileName,  // null -> keep existing
          tourID,
          dayNumber
        },
        transaction: t
      });

      // If no row was updated, INSERT a new one
      const affected =
        (meta && (meta.affectedRows ?? meta.rowCount ?? (Array.isArray(meta) ? meta[1] : 0))) || 0;

      if (!affected) {
        await sequelize.query(`
          INSERT INTO TourDetails (TourID, DayNumber, Photo, Description, CreatedAt)
          VALUES (:tourID, :dayNumber, :photoInsert, :dayDescription, NOW())
        `, {
          replacements: {
            tourID,
            dayNumber,
            photoInsert: savedFileName || 'no_image.jpg',
            dayDescription
          },
          transaction: t
        });
      }
    }

    await t.commit();
    return res.status(200).json({ message: 'Tour updated successfully!' });
  } catch (error) {
    await t.rollback();
    console.error('Error updating tour:', error);
    return res.status(500).json({ error: error.message });
  }
};







module.exports = {
    CreateTour,
    getTheNextId,
    GetAllTourDetails,
    GetAllTourHeader,
    GetAllTourDetailsById,
    UpdateTour,
    deleteTour

}
